#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
netcdf2csv_arealmean_cnst.py — cnst.nc の flon/flat を唯一の位置情報として用い、
セル定義を選択（voronoi）して shapefile と重なり率を計算し、
対象セルの coverage と降水 CSV（全格子・面積加重平均）を出力する。

主な仕様:
  - 位置情報: cnst.nc の flon/flat を使用。
  - セル定義: --cell-def voronoi|dual|primal（default=voronoi）
      * voronoi: 各格子点をサイトとする Voronoi 多角形（等距離直線の分割）。
                  投影空間(EPSG:3857)で有限化＆クリップ＆有効化し、WGS84に逆変換。
      * dual   : 各格子点中心の東西南北“中点”四角形（FCT=499 サンプリング近似）。
      * primal : 通常セル（(j,i)-(j,i+1)-(j+1,i+1)-(j+1,i) の四角形）。
  - 抽出: coverage > --min-coverage のセルを採用。
  - 時刻: netCDF4.num2date で cftime 対応。--shift-hours で補正可能。
  - rain 形状: (time, member, j, i) または (time, j, i) に対応（member=0固定）。
  - 出力: target_coverage.csv は「中心点 (i,j)」をセル名で統一（--index-base で 0/1 始まり）。
  - 追加出力: --export-cells-shp で採用セルのポリゴン出力（shp/gpkg/geojson）。

使い方（例）:
  python netcdf2csv_arealmean_cnst.py ../../hourly/HPB_m009/ -o ./csv_cnst \
    -s ./murayama_hokubu_wgs84.shp --min-coverage 0 \
    --export-cells-shp ./csv_cnst/HPB_m009/selected_cells.shp
"""

import time
import os
import sys
import glob
import argparse
import numpy as np
import pandas as pd
import datetime as dt
from datetime import timedelta
from netCDF4 import Dataset, num2date
import shapefile  # pyshp
from matplotlib.path import Path

# 依存（ある場合のみ使用）
try:
    from scipy.spatial import Voronoi
    _HAVE_SCIPY = True
except Exception:
    _HAVE_SCIPY = False

try:
    import geopandas as gpd
    from shapely.geometry import Polygon, LineString, Point as ShpPoint, MultiPoint
    from shapely.ops import unary_union, transform
    from shapely.errors import TopologicalError
    try:
        from shapely.validation import make_valid  # shapely>=2
        _HAVE_MAKE_VALID = True
    except Exception:
        _HAVE_MAKE_VALID = False
    _HAVE_GPD = True
except Exception:
    _HAVE_GPD = False
    _HAVE_MAKE_VALID = False

try:
    import pyproj
    from pyproj import Transformer
    _HAVE_PYPROJ = True
except Exception:
    _HAVE_PYPROJ = False

from functools import partial

# ------------------------------------------------------------
# ユーティリティ
# ------------------------------------------------------------

def ensure_2d(a, name):
    a = np.asarray(a)
    orig = a.shape
    a = np.squeeze(a)
    while a.ndim > 2:
        a = a[0]
    if a.ndim != 2:
        raise ValueError(f"{name} must be 2D, got {orig} -> {a.shape}")
    return a


def find_rain_files(input_dir: str):
    return sorted(glob.glob(os.path.join(input_dir, "**", "rain.nc"), recursive=True))


def find_cnst_for(rain_nc_path: str, max_up: int = 2) -> str:
    d = os.path.dirname(rain_nc_path)
    for _ in range(max_up + 1):  # 0=同階層, 1=1つ上, ...
        cand = os.path.join(d, "cnst.nc")
        if os.path.exists(cand):
            return cand
        parent = os.path.dirname(d)
        if parent == d:
            break
        d = parent
    raise FileNotFoundError(f"cnst.nc not found within {max_up} levels up from: {rain_nc_path}")


def load_lonlat_from_cnst(cnst_path: str):
    ds = Dataset(cnst_path, "r")
    lon_candidates = ["flon", "lon", "g0_lon", "xx", "x", "longitude"]
    lat_candidates = ["flat", "lat", "g0_lat", "yy", "y", "latitude"]
    lon_name = next((v for v in lon_candidates if v in ds.variables), None)
    lat_name = next((v for v in lat_candidates if v in ds.variables), None)
    if lon_name is None or lat_name is None:
        keys = list(ds.variables.keys())
        ds.close()
        raise RuntimeError(f"cnst.nc に lon/lat 変数が見つかりません: {keys}")
    lon2d = ensure_2d(ds.variables[lon_name][:], lon_name)
    lat2d = ensure_2d(ds.variables[lat_name][:], lat_name)
    ds.close()
    if lon2d.shape != lat2d.shape:
        raise ValueError(f"lon/lat shape mismatch: {lon2d.shape} vs {lat2d.shape}")
    return lon2d, lat2d  # (J, I)


def read_polygon_paths(shp_path: str):
    sf = shapefile.Reader(shp_path, encoding="utf-8", encodingErrors="ignore")
    paths = []
    shapely_polys = []
    for shp in sf.shapes():
        parts = list(shp.parts) + [len(shp.points)]
        for a, b in zip(parts[:-1], parts[1:]):
            pts = np.array(shp.points[a:b])
            if pts.size == 0:
                continue
            if np.max(np.abs(pts[:, 0])) > 360 or np.max(np.abs(pts[:, 1])) > 90:
                raise ValueError("Shapefile の CRS は EPSG:4326 (lon/lat) にしてください。")
            paths.append(Path(pts))
            if _HAVE_GPD:
                try:
                    shapely_polys.append(Polygon(pts))
                except Exception:
                    pass
    if not paths:
        raise ValueError("Shapefile 内に有効なポリゴンが見つかりません。")
    return paths, shapely_polys


def bbox_from_paths(paths):
    lon_min = +1e30; lon_max = -1e30
    lat_min = +1e30; lat_max = -1e30
    for p in paths:
        v = p.vertices
        lon_min = min(lon_min, float(np.min(v[:, 0]))); lon_max = max(lon_max, float(np.max(v[:, 0])))
        lat_min = min(lat_min, float(np.min(v[:, 1]))); lat_max = max(lat_max, float(np.max(v[:, 1])))
    return lon_min, lon_max, lat_min, lat_max


def median_grid_step(a2d, axis):
    a2d = np.asarray(a2d)
    if axis == 1 and a2d.shape[1] > 1:
        dif = np.abs(a2d[:, 1:] - a2d[:, :-1]).ravel()
    elif axis == 0 and a2d.shape[0] > 1:
        dif = np.abs(a2d[1:, :] - a2d[:-1, :]).ravel()
    else:
        return 0.05
    dif = dif[~np.isnan(dif)]
    return float(np.median(dif)) if dif.size else 0.05


def nearest_cell_by_point(center_lon2d, center_lat2d, lat, lon):
    d2 = (center_lat2d - lat) ** 2 + (center_lon2d - lon) ** 2
    j, i = np.unravel_index(np.argmin(d2), d2.shape)
    return np.array([[j, i]])

# ------------------------------------------------------------
# dual coverage（中点四角形 × ポリゴン）
# ------------------------------------------------------------

def coverage_fraction_dual(lon2d, lat2d, sel_jis, paths, FCT=499):
    n = len(sel_jis)
    if n == 0:
        return np.array([], dtype=float)
    if FCT <= 1:
        return np.ones(n, dtype=float)

    J, I = lon2d.shape
    step = 1.0 / FCT
    offs = np.linspace(-0.5 + step / 2, 0.5 - step / 2, FCT)
    uu, vv = np.meshgrid(offs, offs, indexing="xy")
    uv = np.column_stack([uu.ravel(), vv.ravel()])

    fracs = np.empty(n, dtype=float)

    for k, (j, i) in enumerate(sel_jis):
        iW = max(i - 1, 0);   iE = min(i + 1, I - 1)
        jN = max(j - 1, 0);   jS = min(j + 1, J - 1)
        lonE = 0.5 * (lon2d[j, i] + lon2d[j, iE]);   latE = 0.5 * (lat2d[j, i] + lat2d[j, iE])
        lonW = 0.5 * (lon2d[j, iW] + lon2d[j, i]);   latW = 0.5 * (lat2d[j, iW] + lat2d[j, i])
        lonS = 0.5 * (lon2d[jS, i] + lon2d[j, i]);   latS = 0.5 * (lat2d[jS, i] + lat2d[j, i])
        lonN = 0.5 * (lon2d[j, i] + lon2d[jN, i]);   latN = 0.5 * (lat2d[j, i] + lat2d[jN, i])
        C = np.array([[lonE, latE], [lonS, latS], [lonW, latW], [lonN, latN]], dtype=float)
        u = (uv[:, 0] + 0.5); v = (uv[:, 1] + 0.5)
        w0 = (1 - u) * (1 - v); w1 = u * (1 - v); w2 = u * v; w3 = (1 - u) * v
        lon_sub = w0 * C[0, 0] + w1 * C[1, 0] + w2 * C[2, 0] + w3 * C[3, 0]
        lat_sub = w0 * C[0, 1] + w1 * C[1, 1] + w2 * C[2, 1] + w3 * C[3, 1]
        pts = np.column_stack([lon_sub, lat_sub])
        inside = np.zeros(pts.shape[0], dtype=bool)
        for p in paths:
            inside |= p.contains_points(pts)
        fracs[k] = inside.mean()

    return fracs

# ------------------------------------------------------------
# Voronoi（等距離直線の分割セル）ヘルパ
# ------------------------------------------------------------

def _project_xy(lon, lat, lat0=None):
    lon = np.asarray(lon); lat = np.asarray(lat)
    if _HAVE_PYPROJ:
        tf = Transformer.from_crs("EPSG:4326", "EPSG:3857", always_xy=True)
        x, y = tf.transform(lon, lat)
        return np.asarray(x), np.asarray(y)
    if lat0 is None:
        lat0 = float(np.nanmean(lat))
    R = 6371000.0
    x = np.deg2rad(lon) * R * np.cos(np.deg2rad(lat0))
    y = np.deg2rad(lat) * R
    return x, y


def _voronoi_finite_polygons_2d(vor, radius=None):
    if vor.points.shape[1] != 2:
        raise ValueError("Requires 2D input")
    new_regions = []
    new_vertices = vor.vertices.tolist()
    center = vor.points.mean(axis=0)
    if radius is None:
        radius = np.ptp(vor.points, axis=0).max()*2
    all_ridges = {}
    for (p1, p2), (v1, v2) in zip(vor.ridge_points, vor.ridge_vertices):
        all_ridges.setdefault(p1, []).append((p2, v1, v2))
        all_ridges.setdefault(p2, []).append((p1, v1, v2))
    for p1, region_idx in enumerate(vor.point_region):
        vertices = vor.regions[region_idx]
        if all(v >= 0 for v in vertices):
            new_regions.append(vertices); continue
        ridges = all_ridges[p1]
        new_region = [v for v in vertices if v >= 0]
        for p2, v1, v2 in ridges:
            if v2 < 0: v1, v2 = v2, v1
            if v1 >= 0 and v2 >= 0: continue
            t = vor.points[p2] - vor.points[p1]
            t /= np.linalg.norm(t)
            n = np.array([-t[1], t[0]])
            midpoint = vor.points[[p1, p2]].mean(axis=0)
            direction = np.sign(np.dot(midpoint - center, n)) * n
            far_point = vor.vertices[v2] + direction * (radius if radius is not None else 1.0)
            new_vertices.append(far_point.tolist())
            new_region.append(len(new_vertices)-1)
        vs = np.array([new_vertices[v] for v in new_region])
        c = vs.mean(axis=0)
        ang = np.arctan2(vs[:,1]-c[1], vs[:,0]-c[0])
        new_region = [v for _, v in sorted(zip(ang, new_region))]
        new_regions.append(new_region)
    return new_regions, np.asarray(new_vertices)


def build_voronoi_cells(lon2d, lat2d, sel_jis):
    if not _HAVE_SCIPY or not _HAVE_GPD:
        raise RuntimeError("Voronoi requires scipy + shapely/geopandas")
    J, I = lon2d.shape
    lon_flat = lon2d.ravel(); lat_flat = lat2d.ravel()
    x, y = _project_xy(lon_flat, lat_flat)
    vor = Voronoi(np.column_stack([x, y]))
    regions, vertices = _voronoi_finite_polygons_2d(vor)

    # 投影空間で外枠（凸包+余白）を作ってクリップ
    mp_proj = MultiPoint([(xx, yy) for xx, yy in zip(x, y)])
    hull_proj = mp_proj.convex_hull.buffer(20000.0)
    minx, miny, maxx, maxy = hull_proj.bounds
    pad = 50000.0
    bbox_proj = Polygon([(minx-pad, miny-pad), (maxx+pad, miny-pad), (maxx+pad, maxy+pad), (minx-pad, maxy+pad)])
    try:
        hull_proj = hull_proj.intersection(bbox_proj)
    except TopologicalError:
        hull_proj = hull_proj.buffer(0).intersection(bbox_proj)

    if _HAVE_PYPROJ:
        tf_inv = pyproj.Transformer.from_crs("EPSG:3857", "EPSG:4326", always_xy=True)
        to_wgs84 = partial(tf_inv.transform)
    else:
        to_wgs84 = None

    polys = [None]*(J*I)
    for p_idx, reg in enumerate(regions):
        ring_xy = vertices[reg]
        poly_proj = Polygon(ring_xy)
        try:
            poly_proj = poly_proj.intersection(hull_proj)
        except TopologicalError:
            poly_proj = poly_proj.buffer(0).intersection(hull_proj)
        if not poly_proj.is_valid:
            poly_proj = make_valid(poly_proj) if _HAVE_MAKE_VALID else poly_proj.buffer(0)
        poly_ll = transform(to_wgs84, poly_proj) if to_wgs84 is not None else poly_proj
        polys[p_idx] = poly_ll

    out = []
    for j, i in sel_jis:
        out.append(polys[j*I + i])
    return out

# ------------------------------------------------------------
# セルポリゴン出力（任意）
# ------------------------------------------------------------

def export_selected_cell_polys(out_path, cell_def, lon2d, lat2d, sel_jis, cov, index_base=0):
    if not _HAVE_GPD:
        raise RuntimeError("--export-cells-shp requires geopandas/shapely")
    J, I = lon2d.shape
    geoms = []
    recs = []
    if cell_def == "voronoi":
        polys = build_voronoi_cells(lon2d, lat2d, sel_jis)
        geoms = polys
    elif cell_def == "primal":
        for (j, i) in sel_jis:
            i1 = min(i+1, I-1); j1 = min(j+1, J-1)
            ring = [(lon2d[j, i],   lat2d[j, i]),
                    (lon2d[j, i1],  lat2d[j, i1]),
                    (lon2d[j1, i1], lat2d[j1, i1]),
                    (lon2d[j1, i],  lat2d[j1, i]),
                    (lon2d[j, i],   lat2d[j, i])]
            geoms.append(Polygon(ring))
    else:  # dual
        for (j, i) in sel_jis:
            iW = max(i - 1, 0);   iE = min(i + 1, I - 1)
            jN = max(j - 1, 0);   jS = min(j + 1, J - 1)
            E = (0.5 * (lon2d[j, i] + lon2d[j, iE]), 0.5 * (lat2d[j, i] + lat2d[j, iE]))
            S = (0.5 * (lon2d[jS, i] + lon2d[j, i]), 0.5 * (lat2d[jS, i] + lat2d[j, i]))
            W = (0.5 * (lon2d[j, iW] + lon2d[j, i]), 0.5 * (lat2d[j, iW] + lat2d[j, i]))
            N = (0.5 * (lon2d[j, i] + lon2d[jN, i]), 0.5 * (lat2d[j, i] + lat2d[jN, i]))
            ring = [E, S, W, N, E]
            geoms.append(Polygon(ring))
    for (j, i), fr in zip(sel_jis, cov):
        recs.append({
            "i": int(i + (1 if index_base == 1 else 0)),
            "j": int(j + (1 if index_base == 1 else 0)),
            "fraction_area": float(fr)
        })
    gdf = gpd.GeoDataFrame(recs, geometry=geoms, crs="EPSG:4326")
    ext = os.path.splitext(out_path)[1].lower()
    if gdf.empty:
        raise ValueError("GeoDataFrame is empty.")
    
    if ext == ".gpkg":
        gdf.to_file(out_path, layer="cells", driver="GPKG")
    elif ext in (".geojson", ".json"):
        gdf.to_file(out_path, driver="GeoJSON")
    else:
        gdf.to_file(out_path)

# ------------------------------------------------------------
# メイン処理
# ------------------------------------------------------------

def main():
    ap = argparse.ArgumentParser(
        formatter_class=argparse.RawTextHelpFormatter,
        description=(
            "cnst.nc の flon/flat + 指定セル定義で格子選択し、"
            "rain.nc から時系列CSV出力 (全格子・面積加重平均)。"
        ),
    )
    ap.add_argument("inputs", nargs="+", help="入力ディレクトリ（再帰で **/rain.nc を探索）")
    ap.add_argument("-o", "--output", required=True, help="出力ディレクトリ")
    ap.add_argument("-s", "--shapefile", default=None, help="対象ポリゴン shapefile (EPSG:4326)。-p と排他")
    ap.add_argument("-p", "--point", default=None, help="単一点 'lat,lon'（例: 36.03,140.09）。-s と排他")
    ap.add_argument("--shift-hours", type=int, default=0, help="読み込んだ日時を指定時間だけシフト (例: -48)")
    ap.add_argument("--index-base", type=int, choices=[0, 1], default=0, help="target_coverage の i,j を 0/1 始まりで出力 (default=0)")
    ap.add_argument("--cell-def", choices=["voronoi", "dual", "primal"], default="voronoi", help="セル定義を選択")
    ap.add_argument("--min-coverage", type=float, default=1e-12, help="採用する最小被覆率 (default=1e-12). 0 で『少しでも重なれば』")
    ap.add_argument("--export-cells-shp", default=None, help="採用セルの範囲ポリゴンを出力 (shp/gpkg/geojson)")

    args = ap.parse_args()

    if (args.shapefile is None) == (args.point is None):
        print("[ERROR] --shapefile か --point のどちらか一方を指定してください。", file=sys.stderr)
        sys.exit(1)

    input_dirs = sorted(set(args.inputs))
    os.makedirs(args.output, exist_ok=True)

    for in_dir in input_dirs:
        dname = os.path.basename(os.path.normpath(in_dir)) or "out"
        out_dir = os.path.join(args.output, dname)
        os.makedirs(out_dir, exist_ok=True)
        print(f"\ninput: {in_dir}\n  -> output: {out_dir}", flush=True)

        rain_paths = find_rain_files(in_dir)
        if not rain_paths:
            print("  (skip) rain.nc が見つかりません。", flush=True)
            continue

        # 初回: cnst.nc から lon/lat を取得
        try:
            cnst_path = find_cnst_for(rain_paths[0], max_up=2)
        except Exception as e:
            print(f"[ERROR] {e}", file=sys.stderr)
            continue
        lon2d, lat2d = load_lonlat_from_cnst(cnst_path)  # (J,I)
        J, I = lon2d.shape

        # ------------------------- 対象セル抽出 -------------------------
        if args.shapefile:
            print(f"  shapefile: {args.shapefile}")
            shp_paths, shapely_polys = read_polygon_paths(args.shapefile)

            # bbox で広めに候補抽出
            lon_min, lon_max, lat_min, lat_max = bbox_from_paths(shp_paths)
            lon_pad = max(median_grid_step(lon2d, axis=1), 1e-3)
            lat_pad = max(median_grid_step(lat2d, axis=0), 1e-3)
            lon_min -= lon_pad; lon_max += lon_pad
            lat_min -= lat_pad; lat_max += lat_pad

            bbox_mask = ((lon2d >= lon_min) & (lon2d <= lon_max) &
                         (lat2d >= lat_min) & (lat2d <= lat_max))
            jj, ii = np.where(bbox_mask)
            sel_candidates = np.column_stack([jj, ii])  # (m,2) (j,i)

            if sel_candidates.size == 0:
                print("  (skip) bbox 内候補セルが見つかりませんでした。", flush=True)
                continue

            # coverage 計算
            if args.cell_def == "dual":
                cov_all = coverage_fraction_dual(lon2d, lat2d, sel_candidates, shp_paths, FCT=499)
            else:
                if not _HAVE_GPD:
                    raise RuntimeError("cell-def voronoi/primal requires shapely/geopandas installed.")
                cell_polys = []
                if args.cell_def == "primal":
                    for (j, i) in sel_candidates:
                        i1 = min(i+1, I-1); j1 = min(j+1, J-1)
                        ring = [(lon2d[j, i],   lat2d[j, i]),
                                (lon2d[j, i1],  lat2d[j, i1]),
                                (lon2d[j1, i1], lat2d[j1, i1]),
                                (lon2d[j1, i],  lat2d[j1, i]),
                                (lon2d[j, i],   lat2d[j, i])]
                        cell_polys.append(Polygon(ring))
                elif args.cell_def == "voronoi":
                    cell_polys = build_voronoi_cells(lon2d, lat2d, sel_candidates)
                # 被覆率（厳密面積）
                shp_u = unary_union(shapely_polys) if shapely_polys else None
                cov_all = np.zeros(len(sel_candidates), dtype=float)
                for idx, poly in enumerate(cell_polys):
                    try:
                        p = make_valid(poly) if (_HAVE_MAKE_VALID and not poly.is_valid) else (poly if poly.is_valid else poly.buffer(0))
                        if shp_u is not None:
                            inter = p.intersection(shp_u)
                            a = inter.area; A = p.area
                            cov_all[idx] = (a / A) if A > 0 else 0.0
                        else:
                            # フォールバック: 中心点内外
                            j,i = sel_candidates[idx]
                            inside = False
                            for pp in shp_paths:
                                if pp.contains_point((lon2d[j,i], lat2d[j,i])):
                                    inside = True; break
                            cov_all[idx] = 1.0 if inside else 0.0
                    except Exception:
                        cov_all[idx] = 0.0

            mask = cov_all > float(args.min_coverage)
            sel_jis = sel_candidates[mask]
            cov = cov_all[mask]

            if sel_jis.size == 0:
                print("  (skip) coverage>min_coverage のセルがありませんでした。", flush=True)
                continue
        else:
            # 単一点: 最寄りセル1点、被覆率は1.0
            lat_str, lon_str = args.point.split(",")
            sel_jis = nearest_cell_by_point(lon2d, lat2d, float(lat_str), float(lon_str))
            cov = np.ones(len(sel_jis), dtype=float)

        # coverage CSV（中心点の (i,j) をセル名として出力）
        i_out = sel_jis[:, 1] + (1 if args.index_base == 1 else 0)
        j_out = sel_jis[:, 0] + (1 if args.index_base == 1 else 0)
        cov_df = pd.DataFrame({"i": i_out, "j": j_out, "fraction_area": cov})
        cov_df.to_csv(os.path.join(out_dir, "target_coverage.csv"), index=False)
        print(("  [INFO] target_coverage rows: %d  (index_base=%d, min_coverage=%g, cell_def=%s)"
               % (sel_jis.shape[0], args.index_base, args.min_coverage, args.cell_def)))
        print(f"  [INFO] mean fraction: {cov.mean():.6f}, min/max: {cov.min():.6f}/{cov.max():.6f}")

        # セル範囲のポリゴン出力（任意）
        if args.export_cells_shp:
            export_selected_cell_polys(args.export_cells_shp, args.cell_def, lon2d, lat2d, sel_jis, cov, index_base=args.index_base)
            print(f"  [INFO] exported selected cell polygons -> {args.export_cells_shp}")
      
            
      
      
        # break
      
      
      

        # ------------------------- rain 読み出し & 出力 -------------------------
        jmin, imin = sel_jis.min(axis=0)
        jmax, imax = sel_jis.max(axis=0)
        sel_local = sel_jis.copy(); sel_local[:, 0] -= jmin; sel_local[:, 1] -= imin
        col_names = [f"{i}_{j}" for (j, i) in sel_jis]

        all_dates = []
        all_rows = []
        w_sum_all = []
        w_tot_all = []

        for k, f in enumerate(rain_paths, 1):
            print(f"  [{k:3d}/{len(rain_paths)}] {f}", flush=True)
            ds = Dataset(f, "r")

            # 時刻
            tvar = ds.variables["time"]
            cal = getattr(tvar, "calendar", "standard")
            t = num2date(tvar[:], units=tvar.units, calendar=cal)
            if args.shift_hours:
                td = timedelta(hours=args.shift_hours)
                t = [tt - td for tt in t]
            t_str = [pd.Timestamp(str(tt)).strftime("%Y-%m-%d %H:%M:%S") for tt in t]

            # rain 切り出し
            r = ds.variables["rain"]
            if r.ndim == 4:
                sub = r[:, 0, jmin:jmax + 1, imin:imax + 1]  # member=0
            elif r.ndim == 3:
                sub = r[:, jmin:jmax + 1, imin:imax + 1]
            else:
                ds.close()
                raise ValueError(f"想定外の rain 次元: {r.shape}")
            year = int(t_str[0][0:4])
            start_term = dt.datetime(year, 9, 1, 0)
            end_term = dt.datetime(year + 1, 8, 31, 23)

            for ti, ts in enumerate(t_str):
                ts_dt = pd.to_datetime(ts)
                if  (start_term <= ts_dt <= end_term):
                    vals = [float(sub[ti, jloc, iloc]) for (jloc, iloc) in sel_local]
                    all_rows.append(vals)

                    wsum = float(np.dot(vals, cov))
                    wtot = float(cov.sum())
                    w_sum_all.append(wsum)
                    w_tot_all.append(wtot)

                    all_dates.append(ts)

            ds.close()

        if not all_rows:
            print("  (skip) 有効な時刻がありませんでした。", flush=True)
            continue

        df_all = pd.DataFrame(all_rows, columns=col_names)
        df_all.insert(0, "date-hour", all_dates)
        df_all.set_index("date-hour", inplace=True)
        df_all.index = pd.to_datetime(df_all.index)

        arr_mean = np.array(w_sum_all) / np.array(w_tot_all)
        df_mean = pd.DataFrame({"rainfall": np.round(arr_mean, 6)}, index=df_all.index)

        for yr in sorted(df_all.index.year.unique()):
            start = dt.datetime(yr, 9, 1, 0)
            end = dt.datetime(yr + 1, 8, 31, 23)
            d_all = df_all[(df_all.index >= start) & (df_all.index <= end)]
            d_mean = df_mean[(df_mean.index >= start) & (df_mean.index <= end)]
            if not d_all.empty:
                d_all.to_csv(os.path.join(out_dir, f"rain_allgrids_{yr:04d}.csv"))
            if not d_mean.empty:
                d_mean.to_csv(os.path.join(out_dir, f"rain_arealmean_{yr:04d}.csv"))

        print("  -> done.", flush=True)


if __name__ == "__main__":
    try:
        t0 = time.time()
        main()
        t1 = time.time()
        print(f"\nall done. elapsed time: {(t1 - t0)/60:.1f} min", flush=True)
    except KeyboardInterrupt:
        print("Interrupted.", file=sys.stderr)
        sys.exit(130)

